/*:
 * @target MZ
 * @plugindesc [HS] OverlayBlock Sanity After Load v1.0.0 — ロード直後の誤判定だけ抑止（HS_BlockReserveDuringOverlay 用の低リスク回避）
 * @author HS
 *
 * @param graceFramesAfterLoad
 * @text ロード後の猶予フレーム
 * @type number
 * @min 0
 * @max 600
 * @default 90
 * @desc ロード直後、このフレーム数の間だけ「Overlayが開いている判定」を強制falseにします（60fpsで90=約1.5秒）。
 *
 * @param affectOptions
 * @text オプション判定に適用
 * @type boolean
 * @default true
 * @desc HS.isOptionsOverlayOpen() の戻りを、猶予中は false にします。
 *
 * @param affectBacklog
 * @text バックログ判定に適用
 * @type boolean
 * @default false
 * @desc HSBL.isOpen() の戻りを、猶予中は false にします。※通常は不要。誤判定が疑わしい時だけ true。
 *
 * @param debugLog
 * @text デバッグログ
 * @type boolean
 * @default false
 * @desc true の場合、猶予開始/終了などを console に出します。
 *
 * @help
 * ■目的
 * 再起動→ロード直後だけ「Overlayが開いている」と誤判定され、
 * HS_BlockReserveDuringOverlay.js が Sprite_Picture.onClick / reserveCommonEvent を弾いて
 * “特定のピクチャボタン（主にオプション）だけ反応しない”症状が出る場合の低リスク回避です。
 *
 * ■仕組み（重要）
 * - ロード成功のタイミングで猶予カウンタを開始
 * - 猶予カウンタが残っている間だけ
 *    - HS.isOptionsOverlayOpen() を false 固定（affectOptions=true）
 *    - （任意）HSBL.isOpen() を false 固定（affectBacklog=true）
 * これにより、ロード直後の「誤判定」だけを切り落とし、通常時の挙動は維持します。
 *
 * ■導入位置
 * - できるだけ下（特に HS_BlockReserveDuringOverlay.js より下）に置いてください。
 *
 * ■注意
 * - 猶予中に“実際に”オプション/バックログを開いているケースは想定していません
 *   （通常、ロード直後にOverlayが開いた状態になることはありません）。
 */
(() => {
  "use strict";

  const PLUGIN_NAME = document.currentScript.src.split("/").pop().replace(/\.js$/, "");
  const p = PluginManager.parameters(PLUGIN_NAME);

  const GRACE = Math.max(0, Math.min(600, Number(p.graceFramesAfterLoad || 90)));
  const AFFECT_OPT = String(p.affectOptions || "true") === "true";
  const AFFECT_BL  = String(p.affectBacklog || "false") === "true";
  const DEBUG      = String(p.debugLog || "false") === "true";

  const log = (...args) => { if (DEBUG) console.log(`[${PLUGIN_NAME}]`, ...args); };

  let graceLeft = 0;

  function startGrace(reason) {
    if (GRACE <= 0) return;
    graceLeft = GRACE;
    log("grace start:", graceLeft, "frames", "reason:", reason);
  }

  // --- Wrap HS.isOptionsOverlayOpen (if present later or already present)
  function wrapOptionsFn() {
    if (!AFFECT_OPT) return;
    if (!window.HS) window.HS = window.HS || {};
    const HS = window.HS;

    // 二重ラップ防止
    if (HS._hsSanityWrappedOptions) return;
    const orig = HS.isOptionsOverlayOpen;

    HS.isOptionsOverlayOpen = function() {
      if (graceLeft > 0) return false;
      if (typeof orig === "function") return !!orig.apply(this, arguments);
      return false;
    };

    HS._hsSanityWrappedOptions = true;
    log("wrapped: HS.isOptionsOverlayOpen");
  }

  // --- Wrap HSBL.isOpen (if present)
  function wrapBacklogFn() {
    if (!AFFECT_BL) return;
    if (!window.HSBL) window.HSBL = window.HSBL || {};
    const HSBL = window.HSBL;

    if (HSBL._hsSanityWrappedOpen) return;
    const orig = HSBL.isOpen;

    HSBL.isOpen = function() {
      if (graceLeft > 0) return false;
      if (typeof orig === "function") return !!orig.apply(this, arguments);
      return false;
    };

    HSBL._hsSanityWrappedOpen = true;
    log("wrapped: HSBL.isOpen");
  }

  // 起動直後に一度ラップ（HS/HSBL がまだ無くても安全）
  wrapOptionsFn();
  wrapBacklogFn();

  // ロード成功：猶予開始
  if (Scene_Load?.prototype?.onLoadSuccess) {
    const _onLoadSuccess = Scene_Load.prototype.onLoadSuccess;
    Scene_Load.prototype.onLoadSuccess = function() {
      _onLoadSuccess.apply(this, arguments);
      // ここで HS.isOptionsOverlayOpen 等が後から定義される可能性があるので毎回ラップ再試行
      wrapOptionsFn();
      wrapBacklogFn();
      startGrace("onLoadSuccess");
    };
  }

  // extractSaveContents でも猶予開始（ロード経路の違い対策）
  const _extract = DataManager.extractSaveContents;
  DataManager.extractSaveContents = function(contents) {
    _extract.apply(this, arguments);
    wrapOptionsFn();
    wrapBacklogFn();
    startGrace("extractSaveContents");
  };

  // フレームでカウントダウン
  const _SceneBase_update = Scene_Base.prototype.update;
  Scene_Base.prototype.update = function() {
    _SceneBase_update.apply(this, arguments);
    if (graceLeft > 0) {
      graceLeft--;
      if (graceLeft === 0) log("grace end");
    }
  };

  // デバッグ用API
  window.HS = window.HS || {};
  window.HS.isOverlaySanityGraceActive = () => graceLeft > 0;
})();
